<?php

namespace App\Modules\Timetable\Model;

use App\Facades\General;
use App\Models\Tenant;
use App\Modules\Student\Model\StudentClass;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\Facades\DataTables;

class FileManager extends Tenant
{

    use SoftDeletes;

    protected $fillable = ['account_id',
        'class_id',
        'file_title',
        'category',
        'file_name',
        'tags',
        'created_by',
        'updated_by'];

    /**
     * @return BelongsTo
     */
    public function studentClass()
    {
        return $this->belongsTo(StudentClass::class, 'class_id')->withDefault();
    }

    public static function getDatatableFileManager()
    {
        $options           = array();
        $options['object'] = true;

        if (General::isSchoolTeacher()) {
            $options["where"] = ["created_by" => Auth::user()->id];
        }

        if (General::isSchoolStudent()){
            $options["where"] = ["class_id" => Auth::user()->student->class_id];
        }

        $fileManager = new FileManager();
        $model = $fileManager->find($options);

        return DataTables::eloquent($model)
            ->addColumn('download', function ($query) {
                $fileLink = Storage::url('uploads/' . Config::get('account.id') . '/files/'. $query->file_name);
                return '<a href="'.$fileLink.'" target="_blank"><i class="fa fa-download" aria-hidden="true"> Download</i></a>';
            })
            ->addColumn('class', function (FileManager $file) {
                return $file->studentClass->title;
            })
            ->addColumn('date', function ($query) {
                return Carbon::parse($query->created_at)->format('d M Y H:i:a');
            })
            ->addColumn('tags', function ($query) {
                $tagsExploded = explode(',', $query->tags);
                $labelHtml = '';
                foreach($tagsExploded as $tag){
                    $labelHtml.='<button type="button" class="btn yellow btn-outline btn-circle btn-xs m-b-10">'.$tag.'</button>';
                }
                return $labelHtml;
            })
            ->rawColumns(['download', 'tags'])
            ->make(true);
    }

    /**
     * Add / Update the timetable into the resource
     *
     * @param $request
     * @param int $fileManager
     * @return Timetable|Builder|Model|int|object|null
     */
    public static function addUpdateFileManager($request, $fileManager = 0)
    {
        $fileManager = (new FileManager())->findById($fileManager);
        $insert    = false;

        if ($fileManager == null) {
            $fileManager = new FileManager();
            $insert    = true;
        }

        $fileManager->fill($request->all());
        if ($insert == true) {
            $fileManager->created_by = Auth::user()->id;
        } else {
            $fileManager->updated_by = Auth::user()->id;
        }

        $fileManager->account_id = Config::get('account.id');
        $fileManager->save();

        return $fileManager;
    }
}
