<?php

namespace App\Modules\Student\Model;

use App\Facades\General;
use App\Models\Crud;
use App\Models\Tenant;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\Facades\DataTables;

class StudentClass extends Tenant
{
    use SoftDeletes;

    protected $table = 'classes';
    protected $fillable = ['account_id', 'title', 'created_by', 'updated_by'];

    /**
     * @return BelongsTo
     */
    public function student()
    {
        return $this->hasMany('App\Modules\Student\Model\Student', 'class_id')->withDefault();
    }

    /**
     * @return BelongsTo
     */
    /*public function section()
    {
        return $this->hasMany('App\Modules\Student\Model\Section', 'class_id')->withDefault();
    }*/

    /**
     * @return BelongsToMany
     */
    public function section()
    {
        return $this->belongsToMany(Section::class, 'class_sections', 'class_id', 'section_id');
    }

    /**
     * Return student classes
     *
     * @return array|string
     */
    public static function getStudentClasses()
    {
        try {
            $classResults = self::orderBy('title', 'ASC');
            if (General::isSchoolTeacher()) {
                $staffId = Auth::user()->staff->id;
                $classResults->whereIn('id', function ($query) use ($staffId) {
                    $query->select('class_id')->from('class_sections')->where('staff_id', $staffId);
                });
            }
            $classResults = $classResults->get();

            $optionValues = [];
            foreach ($classResults as $classResult) {
                $optionValues[$classResult->id] = $classResult->title;
            }

            return $optionValues;

        } catch (Exception $e) {
            return '';
        }
    }

    /**
     * Return house name nby house id
     *
     * @param $classId
     * @return mixed
     */
    public static function getStudentClassName($classId)
    {
        if (isset($classId) && !empty($classId) && !is_null($classId)) {
            return self::where('id', $classId)->pluck('title')->first();
        } else {
            return null;
        }
    }

    public static function getDatatableClasses()
    {
        $options           = array();
        $options['object'] = true;
        $options['with'] = ['section'];
        $class             = new StudentClass();
        $model             = $class->find($options);

        return DataTables::eloquent($model)
            ->addColumn('action', function (StudentClass $class) {
                return $class->returnAjaxAction('section', $class, ['edit', 'destroy']);
            })
            ->addColumn('sections', function (StudentClass $class) {
                $labelHtml = '';
                foreach($class->section->pluck('title') as $division){
                    $labelHtml.='<button type="button" class="btn yellow btn-outline btn-circle btn-xs m-b-10">'.$division.'</button>';
                }
                return $labelHtml;
            })
            ->rawColumns(['action', 'sections'])
            ->make(true);
    }

    /**
     * Return house name nby house id
     *
     * @param $classId
     * @return mixed
     */
    public  function getStudentClassByTitle($title)
    {
        return self::where('title', $title)->pluck('id')->first();
    }
}
