<?php

namespace App\Modules\Saasuser\Http\Controllers;

use App\Facades\General;
use App\Models\Account;
use App\Modules\Saasuser\Http\Requests\StoreSaasUser;
use App\Modules\Saasuser\Http\Requests\StoreSchool;
use App\Modules\Saasuser\Model\SaasUser;
use Exception;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Session;
use Illuminate\View\View;

class SchoolController extends Controller
{
    /**
     * Module variable
     *
     * @var array
     */
    protected $moduleName;

    /**
     * Class Constructor.
     *
     * @return string
     */
    public function __construct()
    {
        $this->moduleName = $this->getModuleName();
    }

    /**
     * Get Module name.
     *
     * @return string
     */
    public function getModuleName()
    {
        return "saas-school";
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return Factory|JsonResponse|View
     * @throws Exception
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            return Account::getSaasAccounts($this->moduleName);
        }

        General::saasLog('SL000401', ['action_module' => $this->moduleName]);
        return View('saasuser::school.index', ['moduleName' => $this->moduleName]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        $schoolTypes      = Account::getSchoolTypes();
        $schoolCategories = Account::getSchoolCategories();
        return View('saasuser::school.createOrUpdate')->with('moduleName', $this->moduleName)->with("schoolCategories",
            $schoolCategories)
            ->with("schoolTypes", $schoolTypes);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreSchool $request
     * @return RedirectResponse
     */
    public function store(StoreSchool $request)
    {

        // Add student details into user table
        $user = Account::addUpdateSchoolUser($request);

        $account = new Account($request->all());
        if (!isset($request->is_activated)) {
            $account->is_activated = 0;
        }
        $account->user_id = $user->id;
        $account->logo = Account::uploadSchoolLogo($request, 'logo');
        $account->save();

        $user->account_id = $account->id;
        $user->save();

        Session::flash('success',
            trans('locale.success_add_msg', array('module_name' => ucfirst($this->moduleName))));

        General::saasLog('SL000402', [
            'action_module' => $this->moduleName,
            'parent_id'     => $account->id,
            'event_data'    => ['name' => $account->name, 'url' => route($this->moduleName . '.edit', $account->id)]
        ]);

        return redirect()->route($this->moduleName . '.index');
    }

    /**
     * Display the specified resource.
     *
     * @param Account $saasSchool
     * @return Factory|View
     */
    public function show(Account $saasSchool)
    {
        General::saasLog('SL000404', [
            'action_module' => $this->moduleName,
            'parent_id'     => $saasSchool->id,
            'event_data'    => ['name' => $saasSchool->name,
                                'url'  => route($this->moduleName . '.edit', $saasSchool->id)]
        ]);

        return View('saasuser::school.show', compact('saasSchool'))->with('moduleName', $this->moduleName);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param Account $saasSchool
     * @return Factory|View
     */
    public function edit(Account $saasSchool)
    {
        $schoolTypes      = Account::getSchoolTypes();
        $schoolCategories = Account::getSchoolCategories();
        return View('saasuser::school.createOrUpdate', compact('saasSchool'))
            ->with('moduleName', $this->moduleName)->with("schoolCategories", $schoolCategories)
            ->with("schoolTypes", $schoolTypes);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param StoreSchool $request
     * @param Account $saasSchool
     * @return RedirectResponse
     */
    public function update(StoreSchool $request, Account $saasSchool)
    {

        // Add student details into user table
        $user = Account::addUpdateSchoolUser($request, true);

        $saasSchool->fill($request->all());
        if (!isset($request->is_activated)) {
            $saasSchool->is_activated = 0;
        }
        $saasSchool->logo = Account::uploadSchoolLogo($request, 'logo');
        $saasSchool->save();

        Session::flash('success',
            trans('locale.success_update_msg', array('module_name' => ucfirst($this->moduleName))));

        General::saasLog('SL000403', [
            'action_module' => $this->moduleName,
            'parent_id'     => $saasSchool->id,
            'event_data'    => ['name' => $saasSchool->name,
                                'url'  => route($this->moduleName . '.edit', $saasSchool->id)]
        ]);

        return redirect()->route($this->moduleName . '.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Account $saasSchool
     * @return RedirectResponse
     * @throws Exception
     */
    public function destroy(Account $saasSchool)
    {
        General::saasLog('SL000405', [
            'action_module' => $this->moduleName,
            'parent_id'     => $saasSchool->id,
            'event_data'    => ['name' => $saasSchool->name]
        ]);

        Session::flash('success',
            trans('locale.success_delete_msg', array('module_name' => ucfirst($this->moduleName))));
        $saasSchool->delete();

        return redirect()->route($this->moduleName . '.index');
    }
}
