<?php 
ini_set('display_errors', 1);
error_reporting(E_ALL);

include __DIR__ . '/../inc/db.php';
require_once __DIR__ . '/../TCPDF/tcpdf.php';

// =============================
// Helper: Indian Number Format
// =============================
function indian_number_format($num, $decimals = 2) {
    $exploded = explode(".", number_format($num, $decimals, ".", ""));
    $whole = $exploded[0];
    $decimal = isset($exploded[1]) ? $exploded[1] : '';
    $last3 = substr($whole, -3);
    $rest = substr($whole, 0, -3);
    if ($rest != '') {
        $rest = preg_replace("/\B(?=(\d{2})+(?!\d))/", ",", $rest);
        $whole = $rest . "," . $last3;
    }
    return $decimal ? $whole . "." . $decimal : $whole;
}

// =============================
// PDF CLASS
// =============================
class VehicleInvoicePDF extends TCPDF {
    public function Header() {
        $this->Image(__DIR__ . '/../assets/img/logo.png', 12, 10, 40);
        $this->SetFont('dejavusans', '', 8);
        $this->SetXY(140, 10);
        $this->MultiCell(60, 5,
            "GST No: 27ABCDE1234F1Z5\nMobile: +91 98255 59080\nEmail: mytreasuretrip@gmail.com",
            0, 'R', false
        );
        $this->Line(10, 30, 200, 30);
        $this->Ln(5);
        $this->SetFont('dejavusans', 'B', 13);
        $this->SetTextColor(10, 69, 110);
        $this->Cell(0, 23, 'VEHICLE BOOKING INVOICE', 0, 1, 'C');
        $this->SetTextColor(0, 0, 0);
        $this->Line(10, 43, 200, 43);
        $this->Ln(4);
    }

    public function Footer() {
        $this->SetY(-20);
        $this->SetDrawColor(12, 69, 110);
        $this->Line(10, $this->GetY(), 200, $this->GetY());
        $this->Ln(2);
        $this->SetFont('dejavusans', '', 8);
        $this->SetTextColor(80, 80, 80);
        $this->MultiCell(0, 5,
            "My Treasure Trip | 17, Suman Sajni Society, Near Jayant Park, Ghodasar, Ahmedabad\n" .
            "Mobile: +91 98255 59080 | Email: mytreasuretrip@gmail.com | GST: 27ABCDE1234F1Z5",
            0, 'C', false
        );
    }

    public function sectionTitle($title) {
        $this->Ln(3);
        $this->SetFont('dejavusans', 'B', 10);
        $this->SetFillColor(12, 69, 110);
        $this->SetTextColor(255, 255, 255);
        $this->Cell(0, 7, "  " . strtoupper($title), 0, 1, 'L', true);
        $this->SetTextColor(0, 0, 0);
        $this->Ln(2);
    }
}

// =============================
// FETCH BOOKING DATA
// =============================
$booking_id = intval($_GET['id'] ?? 0);
$query = "
SELECT b.*, c.name AS customer_name, c.email AS customer_email, c.phone AS customer_phone, c.address AS customer_address
FROM bookings b
LEFT JOIN customers c ON c.id = b.customer_id
WHERE b.id = $booking_id AND b.booking_type = 'vehicle'
";
$result = $conn->query($query);
$booking = $result->fetch_assoc();

if (!$booking) {
    die("❌ Vehicle booking not found.");
}

// =============================
// CREATE PDF
// =============================
$pdf = new VehicleInvoicePDF();
$pdf->SetMargins(10, 50, 10);
$pdf->AddPage();
$pdf->SetFont('dejavusans', '', 9);
$rupee = '₹';

// =============================
// SECTION: Customer & Booking
// =============================
$pdf->sectionTitle("Customer & Booking Details");

$html = '
<style> td strong { font-weight: bold; color: #0a456e; } </style>
<table cellpadding="5" border="0.5" width="100%">
<tr>
<td width="50%">
<strong>Name:</strong> ' . ucwords($booking['customer_name']) . '<br/>
<strong>Phone:</strong> ' . $booking['customer_phone'] . '<br/>
<strong>Email:</strong> ' . $booking['customer_email'] . '<br/>
<strong>Address:</strong> ' . ucwords($booking['customer_address']) . '
</td>
<td width="50%">
<strong>Booking No:</strong> ' . $booking['booking_code'] . '<br/>
<strong>Booking Date:</strong> ' . date('d-m-Y', strtotime($booking['booking_date'])) . '<br/>
<strong>Status:</strong> ' . ucfirst($booking['status']) . '<br/>
<strong>Payment Status:</strong> ' . ucfirst($booking['payment_status']) . '
</td>
</tr>
</table>';
$pdf->writeHTML($html, true, false, false, false, '');

// =============================
// SECTION: Vehicle Details
// =============================
$pdf->sectionTitle("Vehicle Details");

$vehicle_html = '
<style>
th { background-color:#f0f8ff; color:#0a456e; font-weight:bold; text-align:center; }
</style>
<table cellpadding="5" border="0.5" width="100%">
<tr>
<th>Vehicle Name</th><th>Type</th><th>Passengers</th><th>Pickup</th><th>Drop</th>
</tr>
<tr>
<td align="center">' . strtoupper($booking['vehicle_name'] ?? '-') . '</td>
<td align="center">' . strtoupper($booking['vehicle_type'] ?? '-') . '</td>
<td align="center">' . ($booking['vehicle_passengers'] ?? '-') . '</td>
<td align="center">' . ucwords($booking['vehicle_pickup_location']) . '</td>
<td align="center">' . ucwords($booking['vehicle_drop_location']) . '</td>
</tr>
<tr>
<td colspan="5"><strong>Pickup Date & Time:</strong> ' . date('d-m-Y h:i A', strtotime($booking['vehicle_pickup_datetime'])) . '</td>
</tr>
<tr>
<td colspan="5"><strong>Return Date & Time:</strong> ' . 
    ($booking['vehicle_return_datetime'] ? date('d-m-Y h:i A', strtotime($booking['vehicle_return_datetime'])) : '-') . '</td>
</tr>
<tr>
<td colspan="3"><strong>Total KM:</strong> ' . $booking['vehicle_total_km'] . '</td>
<td colspan="2"><strong>Rate / KM:</strong> '.$rupee.' ' . indian_number_format($booking['vehicle_rate_per_km']) . '</td>
</tr>
</table>';
$pdf->writeHTML($vehicle_html, true, false, false, false, '');

// =============================
// SECTION: Payment Details
// =============================
$pdf->sectionTitle("Payment Summary");

$base_fare = floatval($booking['transport_charges']);
$driver_allowance = floatval($booking['vehicle_driver_allowance']);
$extra = floatval($booking['extra_charges']);
$discount = floatval($booking['discount_amount']);
$gst_rate = floatval($booking['gst_rate']);
$gst = floatval($booking['gst_amount']);
$total = floatval($booking['total_amount']);
$paid = floatval($booking['paid_amount']);
$balance = floatval($booking['balance_amount']);

$pricing_html = '
<table cellpadding="5" border="0.5" width="100%">
<tr><td><strong>Base Fare</strong></td><td align="right">'.$rupee.' '.indian_number_format($base_fare).'</td></tr>';

if ($driver_allowance > 0)
    $pricing_html .= '<tr><td><strong>Driver Allowance</strong></td><td align="right">'.$rupee.' '.indian_number_format($driver_allowance).'</td></tr>';

if ($extra > 0)
    $pricing_html .= '<tr><td><strong>Extra Charges</strong></td><td align="right">'.$rupee.' '.indian_number_format($extra).'</td></tr>';

if ($discount > 0)
    $pricing_html .= '<tr><td><strong>Discount</strong></td><td align="right">-'.$rupee.' '.indian_number_format($discount).'</td></tr>';

if ($gst > 0)
    $pricing_html .= '<tr><td><strong>GST ('.$gst_rate.'%)</strong></td><td align="right">'.$rupee.' '.indian_number_format($gst).'</td></tr>';

$pricing_html .= '
<tr style="background-color:#f9f9f9; font-weight:bold; color:#0a456e;">
    <td>Total Amount</td><td align="right">'.$rupee.' '.indian_number_format($total).'</td>
</tr>
<tr><td><strong>Paid Amount</strong></td><td align="right">'.$rupee.' '.indian_number_format($paid).'</td></tr>
<tr style="font-weight:bold; color:#0a456e;">
    <td>Balance Amount</td><td align="right">'.$rupee.' '.indian_number_format($balance).'</td>
</tr>
</table>';
$pdf->writeHTML($pricing_html, true, false, false, false, '');

// =============================
// OUTPUT PDF
// =============================
$pdf->Output('vehicle_invoice_' . $booking['booking_code'] . '.pdf', 'I');
?>
