<?php

namespace App\Modules\Timetable\Http\Controllers;

use App\Facades\General;
use App\Modules\Staff\Model\Staff;
use App\Modules\Student\Model\Section;
use App\Modules\Student\Model\StudentClass;
use App\Modules\Student\Model\Subject;
use App\Modules\Student\Model\Student;
use App\Modules\Timetable\Model\FileManager;
use App\Modules\Timetable\Model\Timetable;
use App\Modules\Timetable\Requests\StoreFile;
use App\Modules\Timetable\Requests\StoreTimetable;
use Exception;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Session;
use Illuminate\View\View;

class FileController extends Controller
{
    /**
     * Module variable
     *
     * @var array
     */
    protected $moduleName;

    /**
     * Class Constructor.
     *
     * @return string
     */
    public function __construct()
    {
        $this->moduleName = $this->getModuleName();
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     * @throws Exception
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            return FileManager::getDatatableFileManager();
        }
        $data['studentClasses'] = StudentClass::getStudentClasses();
        $data['fileCategory']        = General::getFileCategory();

        General::userLog('SL001001', ['action_module' => $this->moduleName]);

        return View('timetable::file.index', ['moduleName' => $this->moduleName, 'data' => $data]);
    }

    /**
     * Upload file on the server
     *
     * @param Request $request
     * @return false|string
     */
    public function uploadFile(Request $request){

        if ($request->isMethod("post")) {

            $fileResponse = General::uploadFileManager($request, 'file', 'files');

            return json_encode(["success" => true, "data" => $fileResponse['filename']]);
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return Response
     */
    public function create()
    {
        return View('timetable::createOrUpdate')->with('moduleName', $this->moduleName);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreTimetable $request
     * @return JsonResponse
     */
    public function store(StoreFile $request)
    {

        if ($request->ajax()) {
            // Add student details into student table
            $fileManager = FileManager::addUpdateFileManager($request);

            Session::flash('success',
                trans('locale.success_add_msg', array('module_name' => ucfirst($this->moduleName))));

            General::userLog('SL001002', [
                'action_module' => $this->moduleName,
                'parent_id'     => $fileManager->id,
                'event_data'    => ['name' => $fileManager->class_id,
                                    'url'  => route($this->moduleName . '.edit', $fileManager->id)]
            ]);

            return response()->json(["status" => 'added', "status_code" => 200], 200);
        }

        return response()->json(["status" => 'Bad request', "status_code" => 401], 401);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param $fileManager
     * @return JsonResponse
     */
    public function edit($fileManager)
    {
        if (request()->ajax()) {
            $data = Timetable::findOrFail($fileManager);

            return response()->json(['result' => $data], 200);
        }

        return response()->json(['result' => 'Bad request'], 401);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param Request $request
     * @param $fileManager
     * @return JsonResponse
     */
    public function update(Request $request, $fileManager)
    {
        if ($request->ajax()) {

            // update file details into storage
            $fileManager = FileManager::addUpdateFileManager($request, $fileManager);

            General::userLog('SL001002', [
                'action_module' => $this->moduleName,
                'parent_id'     => $fileManager->id,
                'event_data'    => ['name' => $fileManager->class_id,
                                    'url'  => route($this->moduleName . '.edit', $fileManager->id)]
            ]);

            return response()->json(["status" => 'updated'], 200);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Student $student
     * @return RedirectResponse
     * @throws Exception
     */
    public function destroy($student)
    {
        $student = (new Student())->findById($student);
        General::userLog('SL001005', [
            'action_module' => $this->moduleName,
            'parent_id'     => $student->id,
            'event_data'    => ['name' => $student->user->name]
        ]);

        Session::flash('success',
            trans('locale.success_delete_msg', array('module_name' => ucfirst($this->moduleName))));
        $student->delete();

        return redirect()->route($this->moduleName . '.index');
    }

    public function studyMaterial(Request $request){
        if ($request->ajax()) {
            return FileManager::getDatatableFileManager();
        }

        General::userLog('SL001001', ['action_module' => $this->moduleName]);

        return View('timetable::file.study_material', ['moduleName' => $this->moduleName]);
    }

    /**
     * Get Module name.
     *
     * @return string
     */
    public function getModuleName()
    {
        return "file_manager";
    }
}
