<?php

namespace App\Modules\Student\Http\Controllers;

use App\Facades\General;
use App\Modules\Student\Model\Section;
use App\Modules\Student\Model\StudentClass;
use App\Modules\Student\Requests\StoreClass;
use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Modules\Student\Http\Requests\StoreStudent;

class ClassController extends Controller
{
    /**
     * Module variable
     *
     * @var array
     */
    protected $moduleName;

    /**
     * Class Constructor.
     *
     * @return string
     */
    public function __construct()
    {
        $this->moduleName = $this->getModuleName();
    }

    /**
     * Display a listing of the resource.
     *
     * @param Request $request
     * @return JsonResponse
     * @throws Exception
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            return StudentClass::getDatatableClasses();
        }
        General::userLog('SL000601', ['action_module' => $this->moduleName]);
        return View('student::class.index', ['moduleName' => $this->moduleName]);
    }

    /**
     * Crud method of section
     *
     * @param StoreClass $request
     * @param int $studentClassId
     * @return JsonResponse
     */
    public function crud(StoreClass $request, $studentClassId = 0)
    {
        $studentClass = (new StudentClass())->findById($studentClassId);
        $insert  = false;
        $activityId = null;
        if ($studentClass == null) {
            $studentClass = new StudentClass();
            $insert  = true;
        }

        $studentClass->fill($request->all());
        $studentClass->account_id = General::getAccountId();
        if ($insert == true) {
            $studentClass->created_by = General::getUserId();
            $activityId = 'SL000602';
        } else {
            $studentClass->updated_by = General::getUserId();
            $activityId = 'SL000603';
        }
        $studentClass->save();
        if ($studentClass) {
            // $studentClass->section()->detach();
            $studentClass->section()->sync($request->get('divisions'));
        }

        General::userLog($activityId, [
            'action_module' => $this->moduleName,
            'parent_id'     => $studentClass->id,
            'event_data'    => ['name' => $studentClass->title, 'url' => route($this->moduleName.'.edit', $studentClass->id)]
        ]);

        return response()->json(["status" => 'added'], 200);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param $classId
     * @return JsonResponse
     * @throws \Throwable
     */
    public function edit($classId)
    {
        if (request()->ajax()) {
            $data = StudentClass::findOrFail($classId);
            $gradeDivisions = '';

            if ($data->section){
                $divisionArr = $data->section->pluck('pivot')->pluck('section_id')->toArray();
                $gradeDivisions = Section::getDivisions($divisionArr);
            }

            return response()->json(['result' => $data, 'gradeDivisions' => $gradeDivisions]);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param $classId
     * @return JsonResponse
     * @throws Exception
     */
    public function destroy($classId)
    {
        if (request()->ajax()) {
            $studentClass = (new StudentClass())->findById($classId);
            General::userLog('SL000605', [
                'action_module' => $this->moduleName,
                'parent_id'     => $studentClass->id,
                'event_data'    => ['name' => $studentClass->title]
            ]);
            $studentClass->delete();

            return response()->json(['msg' => trans('locale.success_delete_msg')]);
        }
    }

    /**
     * Get Module name.
     *
     * @return string
     */
    public function getModuleName()
    {
        return "class";
    }

    /**
     * Return the render options of section
     * for particular class
     *
     * @param StudentClass $classId
     * @return array|string
     * @throws \Throwable
     */
    public function onChangeClass($classId)
    {
        $classId = (new StudentClass())->findById($classId);
        $optionValues = [];
        foreach ($classId->section as $classSection) {
            $section                    = Section::getSectionById($classSection->pivot->section_id);
            $optionValues[$section->id] = $section->title;
        }
        asort($optionValues);

        $classSections = view('student::class.dropdown.classSections', compact('optionValues'))->render();

        return response()->json(["classSections" => $classSections], 200);
    }

    public function getDivisions(){
        $gradeDivisions = Section::getDivisions();

        return response()->json(["gradeDivisions" => $gradeDivisions], 200);
    }
}
